/* Classify protein sequences
 *
 * Copyright 2014 Peter Meinicke, Robin Martinjak
 *
 * This file is part of libuproc.
 *
 * libuproc is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * libuproc is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with libuproc.  If not, see <http://www.gnu.org/licenses/>.
 */

#define _GNU_SOURCE

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdbool.h>
#include <string.h>

#include "uproc/common.h"
#include "uproc/error.h"
#include "uproc/bst.h"
#include "uproc/list.h"
#include "uproc/protclass.h"

int PRINTFLAG = 0;


typedef struct tax_scores_s tax_scores;

tax_scores *
uproc_tax_create(int xDim, int yDim)
{
tax_scores *return_val;
int i,j;

int nMaxSeq = 10000;

return_val = malloc(sizeof(*return_val));
if( return_val == NULL )
  fprintf(stderr, "Cannot allocate return_val");
return_val->listsize = xDim;
return_val->seqLength = -1;
return_val->seqLength2 = 0;
return_val->used = 0;
return_val->words = 0;
return_val->words_perfect = 0;
return_val->seen_X = malloc(sizeof (return_val->seen_X) * xDim);
if( return_val->seen_X == NULL )
  fprintf(stderr, "Cannot allocate seen_X");
 
return_val->scoreMat = malloc(sizeof (*return_val->scoreMat) * xDim);
if( return_val->scoreMat == NULL )
  fprintf(stderr, "Cannot allocate scoreMat");
   for (i=0; i<xDim; i++)
   {
   return_val->scoreMat[i] = malloc(sizeof (return_val->scoreMat) * yDim);   
   if( return_val->scoreMat[i] == NULL )
  fprintf(stderr, "Cannot allocate row %d", i);

     for(j=0; j<yDim; j++)
     {
     return_val->scoreMat[i][j] = 0.0;
     }
   }

return_val->countMat = malloc(sizeof (*return_val->countMat) * xDim);

if( return_val->countMat == NULL )
  fprintf(stderr, "Cannot allocate countMat");

   for (i=0; i<xDim; i++)
   {
   return_val->countMat[i] = malloc(sizeof (return_val->countMat) * yDim);

   if( return_val->countMat[i] == NULL )
   fprintf(stderr, "Cannot allocate row %d", i); 
   
     for(j=0; j<yDim; j++)
     {
     return_val->countMat[i][j] = 0;
     }
   }
   
return(return_val);  
};

int 
uproc_tax_find(uproc_family val, uproc_family *plist, int max)
{
int i;
  for (i=0; i<max; i++)
  {
    if (val == plist[i])
    {
    return(i); 
    }
    
  }
return(-1);
}


void uproc_tax_addscore(uproc_family xDim, uproc_family yDim, double **pArray, int **pArray2, double val)
{
  //HACK
uproc_family i;
//pArray[xDim][yDim] += val;
//pArray2[xDim][yDim]++;
  for (i = yDim; i<9; i++)
  {
  pArray[xDim][i] += val;
  pArray2[xDim][i]++;
  
  }

}

int uproc_tax_result(double **pArray)
{
  
}

struct uproc_protclass_s
{
    enum uproc_protclass_mode mode;
    const uproc_substmat *substmat;
    const uproc_ecurve *fwd;
    const uproc_ecurve *rev;
    uproc_protfilter *filter;
    void *filter_arg;
    struct uproc_protclass_trace
    {
        uproc_protclass_trace_cb *cb;
        void *cb_arg;
    } trace;
};

/*********************
 * score computation *
 *********************/

struct sc
{
    size_t index;
    double total, dist[UPROC_WORD_LEN];
};

static void
reverse_array(void *p, size_t n, size_t sz)
{
    unsigned char *s = p, tmp;
    size_t i, k, i1, i2;

    for (i = 0; i < n / 2; i++) {
        for (k = 0; k < sz; k++) {
            i1 = sz * i + k;
            i2 = sz * (n - i - 1) + k;
            tmp = s[i1];
            s[i1] = s[i2];
            s[i2] = tmp;
        }
    }
}

static void
sc_init(struct sc *s)
{
    size_t i;
    s->index = -1;
    s->total = 0.0;
    for (i = 0; i < UPROC_WORD_LEN; i++) {
        s->dist[i] = -INFINITY;
    }
}

static void
sc_add(struct sc *score, size_t index, double dist[static UPROC_SUFFIX_LEN],
       bool reverse)
{
    size_t i, diff;
    double tmp[UPROC_WORD_LEN];

    for (i = 0; i < UPROC_PREFIX_LEN; i++) {
        tmp[i] = -INFINITY;
    }
    memcpy(tmp + UPROC_PREFIX_LEN, dist, sizeof *dist * UPROC_SUFFIX_LEN);
    if (reverse) {
        reverse_array(tmp, UPROC_WORD_LEN, sizeof *tmp);
    }

    if (score->index != (size_t) -1) {
        diff = index - score->index;
        if (diff > UPROC_WORD_LEN) {
            diff = UPROC_WORD_LEN;
        }
        for (i = 0; i < diff; i++) {
            if (isfinite(score->dist[i])) {
                score->total += score->dist[i];
                score->dist[i] = -INFINITY;
            }
        }
    }
    else {
        diff = 0;
    }

    for (i = 0; i + diff < UPROC_WORD_LEN; i++) {
#define MAX(a, b) (a > b ? a : b)
        score->dist[i] = MAX(score->dist[i + diff], tmp[i]);
    }
    for (; i < UPROC_WORD_LEN; i++) {
        score->dist[i] = tmp[i];
    }
    score->index = index;
}

static double
sc_finalize(struct sc *score)
{
    size_t i;
    for (i = 0; i < UPROC_WORD_LEN; i++) {
        if (isfinite(score->dist[i])) {
            score->total += score->dist[i];
        }
    }
    return score->total;
}

static int
scores_add(uproc_bst *scores, uproc_family family, uproc_family xDim, uproc_family yDim, size_t index,
           double dist[static UPROC_SUFFIX_LEN], bool reverse)
{
    struct sc sc;
    union uproc_bst_key key = { .uint = family };
    sc_init(&sc);
    (void) uproc_bst_get(scores, key, &sc, &xDim, &yDim);
    //printf("%d\t%u\t%u\n",key.uint,xDim,yDim);
    
    sc_add(&sc, index, dist, reverse);
    return uproc_bst_update(scores, key, &sc, &xDim, &yDim);
}


static int
scores_add_word(tax_scores *taxp, const uproc_protclass *pc, uproc_bst *scores,
                const struct uproc_word *word,
                size_t index, bool reverse, const uproc_ecurve *ecurve,
                const uproc_substmat *substmat)
{
    int res;
    struct uproc_word
        lower_nb = UPROC_WORD_INITIALIZER,
        upper_nb = UPROC_WORD_INITIALIZER;
    uproc_family lower_family, lower_xDim, lower_yDim, upper_family, upper_xDim, upper_yDim;
    double dist[UPROC_SUFFIX_LEN];
    double tmp_wordscore;
    int iter;
    if (!ecurve) {
        return 0;
    }
    uproc_ecurve_lookup(ecurve, word, &lower_nb, &lower_family, &lower_xDim, &lower_yDim, &upper_nb,
                        &upper_family, &upper_xDim, &upper_yDim);
    
    uproc_substmat_align_suffixes(substmat, word->suffix, lower_nb.suffix, dist);
    tmp_wordscore = 0;
      for (iter = 0; iter < UPROC_SUFFIX_LEN; iter++)
      {
      tmp_wordscore += dist[iter];
      }
      
    if (pc->trace.cb) {
        pc->trace.cb(&lower_nb, lower_family, lower_xDim, lower_yDim, index, reverse, dist,
                     pc->trace.cb_arg);
    }
    int r_val;
    r_val = -1;
    
    //printf("4)\t%u\t%u\t%u\t%f\t\n",lower_family,lower_xDim,lower_yDim,tmp_wordscore); 
    
    if (tmp_wordscore > 0)
    {
      
      if (taxp->seqLength == -1)
      {
      taxp->seqLength2 = 18;
      }
      
      else
      {
      int val = index - taxp->seqLength;
	if (val > 18)
	{
	taxp->seqLength2 += 18;  
	}
	
	else
	{
	taxp->seqLength2 += val; 
	}
      }  
      taxp->seqLength = index;
      
    //char stringstr[UPROC_WORD_LEN];
    //uproc_word_to_string(stringstr, word, alpha);
    
    
    	if (lower_nb.suffix == word->suffix)
	{
	taxp->words_perfect++;
	}
    
      if (PRINTFLAG == 3)
      {
      printf("4)\t%u\t%u\t%u\t%f\t\t%lu\t",lower_family,lower_xDim,lower_yDim,tmp_wordscore,lower_nb.suffix);
	if (lower_nb.suffix == word->suffix)
	{
	taxp->words_perfect++;
	printf("1\t");
	}
	
	else
	{
	printf("0\t");
	}
	
	if (lower_nb.prefix == word->prefix)
	{
	printf("1\n");
	}
	
	else
	{
	printf("0\n");
	}
      }
    taxp->words++;
    
      if (taxp->used == 0)
      {
// 	for (iter = 0; iter<9; iter++)
// 	{
// 	printf("%d\t",*taxp->countMat[iter]);
// 	}
// 	printf("\n");
      taxp->seen_X[taxp->used] = lower_xDim;
      r_val = taxp->used;
      taxp->used++;
      }
      
      else
      {
//HACK   
	if (taxp->used < 4000)
	{
	  r_val = uproc_tax_find(lower_xDim, taxp->seen_X, taxp->used);
	    if (r_val == -1)
	    {
	    taxp->seen_X[taxp->used] = lower_xDim;
	    r_val = taxp->used;
	    taxp->used++;
	    
	    }
	}
//HACK   
      }
      //printf("\tadd score tax.\t%d\n",r_val);
      uproc_tax_addscore(r_val, lower_yDim, taxp->scoreMat, taxp->countMat, tmp_wordscore); 
    }
      
    
    
    //printf("1)\t%u\t%u\t%u\tScore: %f\t\t%d\t%d\n",lower_family,lower_xDim,lower_yDim,tmp_wordscore, taxp->used, r_val);
    res = scores_add(scores, lower_family, lower_xDim, lower_yDim, index, dist, reverse);
    
//    tmp_wordscore = 0;
//     for (iter = 0; iter < UPROC_SUFFIX_LEN; iter++)
//      {
//      tmp_wordscore += dist[iter];
//      }  
      
    
    if (res || !uproc_word_cmp(&lower_nb, &upper_nb)) {
        return res;
    }
    uproc_substmat_align_suffixes(substmat, word->suffix, upper_nb.suffix,
                                  dist);
     
    //printf("2)\t%u\t%u\t%u\tScore: %f\n",lower_family,lower_xDim,lower_yDim,tmp_wordscore);
    if (pc->trace.cb) {
        pc->trace.cb(&upper_nb, upper_family, upper_xDim, upper_yDim, index, reverse, dist,
                     pc->trace.cb_arg);
    }
    res = scores_add(scores, upper_family, upper_xDim, upper_yDim, index, dist, reverse);;
    tmp_wordscore = 0;
      for (iter = 0; iter < UPROC_SUFFIX_LEN; iter++)
      {
      tmp_wordscore += dist[iter];
      }  
    
    
    
    if (tmp_wordscore > 0)
    {
      if (taxp->seqLength == -1)
      {
      taxp->seqLength2 = 18;
      }
      
      else
      {
      int val = index - taxp->seqLength;
	if (val > 18)
	{
	taxp->seqLength2 += 18;  
	}
	
	else
	{
	taxp->seqLength2 += val; 
	}
      }   
   taxp->seqLength = index;
      
	if (upper_nb.suffix == word->suffix)
	{
	taxp->words_perfect++;
	}
      
      
    //char stringstr[UPROC_WORD_LEN];
    //uproc_word_to_string(stringstr, word, alpha);
    if (PRINTFLAG == 3)
    {
    printf("5)\t%u\t%u\t%u\t%f\t\t%lu\t",upper_family,upper_xDim,upper_yDim,tmp_wordscore,upper_nb.suffix);
	if (upper_nb.suffix == word->suffix)
	{
	printf("1\n");
	taxp->words_perfect++;
	}
	
	else
	{
	printf("0\t");
	}
	
	if (upper_nb.prefix == word->prefix)
	{
	printf("1\n");
	}
	
	else
	{
	printf("0\n");
	}      
    }

      
    taxp->words++;
    r_val = -1;
      if (taxp->used == 0)
      {
      taxp->seen_X[taxp->used] = upper_xDim;
      r_val = taxp->used;
      taxp->used++;
      }
      
      else
      {
//HACK      
	if (taxp->used < 4000)
	{
	  r_val = uproc_tax_find(upper_xDim, taxp->seen_X, taxp->used);
	    if (r_val == -1)
	    {
	    taxp->seen_X[taxp->used] = upper_xDim;
	    r_val = taxp->used;
	    taxp->used++;
	    }
	}
      }
//HACK   
    //printf("\tadd score tax.\t%d\n",r_val);
    uproc_tax_addscore(r_val, upper_yDim, taxp->scoreMat, taxp->countMat, tmp_wordscore);
    //printf("\tdone.\n");
    
    }
    
    return res;
}

static int
scores_compute(tax_scores *taxp,const struct uproc_protclass_s *pc, const char *seq,
               uproc_bst *scores)
{
    int res;
    uproc_worditer *iter;
    size_t index;
    struct uproc_word
        fwd_word = UPROC_WORD_INITIALIZER,
        rev_word = UPROC_WORD_INITIALIZER;

    iter = uproc_worditer_create(seq, uproc_ecurve_alphabet(pc->fwd));
    if (!iter) {
        return -1;
    }

    while (res = uproc_worditer_next(iter, &index, &fwd_word, &rev_word),
           !res)
    {
	//printf("%lu\tfwd.\n",index);
        res = scores_add_word(taxp, pc, scores, &fwd_word, index, false, pc->fwd,
                              pc->substmat);
        if (res) {
            break;
        }
	//printf("%lu\trev.\n",index);
        res = scores_add_word(taxp, pc, scores, &rev_word, index, true, pc->rev,
                              pc->substmat);
        if (res) {
            break;
        }
    }
    uproc_worditer_destroy(iter);
    return res == -1 ? -1 : 0;
}


/****************
 * finalization *
 ****************/

static int
scores_finalize(const struct uproc_protclass_s *pc, const char *seq,
                uproc_bst *score_tree, uproc_list *results, int bTax[])
{
    int res = 0;
    uproc_bstiter *iter;
    union uproc_bst_key key;
    struct sc value;
    size_t seq_len = strlen(seq);
    struct uproc_protresult pred, pred_max = { .score = -INFINITY };

    uproc_family xDim, yDim;
    
    iter = uproc_bstiter_create(score_tree);
    if (!iter) {
        return -1;
    }
    while (!uproc_bstiter_next(iter, &key, &value, &xDim, &yDim)) {
        uproc_family family = key.uint;
	//printf("trash\t%u\t%u\t%u\n",family,xDim,yDim);
        double score = sc_finalize(&value);
        if (pc->filter &&
            !pc->filter(seq, seq_len, family, score, pc->filter_arg)) {
            continue;
        }
        
        pred.score = score;
        pred.family = family;
	pred.taxa = bTax[0];
	pred.xDim = bTax[1];
	pred.yDim = bTax[2];
	//printf("\ttrash\t%u\t%f\t%u\t%u\n",pred.family,pred.score,bTax[1],bTax[2]);
        if (pc->mode == UPROC_PROTCLASS_MAX) {
            if (!uproc_list_size(results)) {
                pred_max = pred;
                res = uproc_list_append(results, &pred);
                if (res) {
                    break;
                }
            }
            else if (pred.score > pred_max.score) {
                pred_max = pred;
                uproc_list_set(results, 0, &pred_max);
            }
        }
        else {
            uproc_list_append(results, &pred);
        }
    }
    uproc_bstiter_destroy(iter);
    return res;
}


/**********************
 * exported functions *
 **********************/

uproc_protclass *
uproc_protclass_create(enum uproc_protclass_mode mode, const uproc_ecurve *fwd,
                       const uproc_ecurve *rev, const uproc_substmat *substmat,
                       uproc_protfilter *filter, void *filter_arg)
{
    struct uproc_protclass_s *pc;
    if (!(fwd || rev)) {
        uproc_error_msg(UPROC_EINVAL,
                        "protein classifier requires at least one ecurve");
        return NULL;
    }
    pc = malloc(sizeof *pc);
    if (!pc) {
        uproc_error(UPROC_ENOMEM);
        return NULL;
    }
    *pc = (struct uproc_protclass_s) {
        .mode = mode,
        .substmat = substmat,
        .fwd = fwd,
        .rev = rev,
        .filter = filter,
        .filter_arg = filter_arg,
        .trace = {
            .cb = NULL,
            .cb_arg = NULL,
        },
    };
    return pc;
}

void
uproc_protclass_destroy(uproc_protclass *pc)
{
    free(pc);
}


static void
map_list_protresult_free(void *value, void *opaque)
{
    (void) opaque;
    uproc_protresult_free(value);
}


int
uproc_protclass_classify(const uproc_protclass *pc, const char *seq,
                         uproc_list **results, unsigned int **TaxArray, tax_scores *taxS)
{
    int res;
    uproc_bst *scores;
    int i,j;
    
//STUFF    
    
    if (!*results) {
        *results = uproc_list_create(sizeof (struct uproc_protresult));
        if (!*results) {
            return -1;
        }
    }
    else {
        uproc_list_map(*results, map_list_protresult_free, NULL);
        uproc_list_clear(*results);
    }
    
    scores = uproc_bst_create(UPROC_BST_UINT, sizeof (struct sc));
    if (!scores) {
        return -1;
    }
    res = scores_compute(taxS, pc, seq, scores);
    
    if (taxS->used > 0)
    {
    int ListVec[taxS->used];
      if (res || uproc_bst_isempty(scores)) {
	  goto error;
      }
    
    double ScoreMax[9],TotalScoreMax[9],Score2ndMax[9];
    
    
    double FirstScore[9],SecondScore[9];
    int First_X[9],Second_X[9];
    int FirstWords[9],SecondWords[9];
    
    int TaxVec[9];
      for (j=0; j<9; j++)
      {
      int uniqueVec[taxS->used];
      int uVec_X[taxS->used];
      int uPos = 0;
      int k;
      
      ScoreMax[j]=0;
      TotalScoreMax[j]=0;
      Score2ndMax[j] = 0;
      
      FirstScore[j] = 0.0;
      SecondScore[j] = 0.0;
      FirstWords[j] = 0;
      SecondWords[j] = 0;
      
      First_X[j] = 0;
      Second_X[j] = 0;

      
      
      
	for (i=0; i<taxS->used; i++)
	{
	int found = 0;
	//printf("%d\t%f\n",TaxArray[taxS->seen_X[i]][j],taxS->scoreMat[i][j]);
	  for (k = 0; k < uPos; k++)
	  {
	    if (uniqueVec[k] == TaxArray[taxS->seen_X[i]][j])
	    {
	    found = 1;
	    uVec_X[k] = taxS->seen_X[i];
	    break;
	    }
	  }
	  
	  if (found == 0)
	  {
	  uniqueVec[uPos] = TaxArray[taxS->seen_X[i]][j];
	  //printf("%d\tNew TaxID:\t%d\tSeen: %d\t i: %d\n",j,uniqueVec[uPos],taxS->seen_X[i],i);
	  uVec_X[uPos] = taxS->seen_X[i];
	  //HACK

	  
	  uPos++; 
	  }
	}
      //printf("yDim: %d\tuPos: %d\n", j, uPos);
      
      
      //sum up scores of same entry
      double ScoreVec[uPos];
      int CountVec[uPos];
      
	for (i=0; i<uPos; i++)
	{
	ScoreVec[i] = 0.0;
	CountVec[i] = 0;
	  for (k=0; k<taxS->used; k++)
	  {
	    if (uniqueVec[i] == TaxArray[taxS->seen_X[k]][j])
	    {
	      //printf(">\t%d\t%f\n",uniqueVec[i],taxS->scoreMat[k][j]);
	      ScoreVec[i] += taxS->scoreMat[k][j];
	      CountVec[i] += taxS->countMat[k][j];
	      
	      
		if (ScoreMax[j] < taxS->scoreMat[k][j])
		{
		uVec_X[i] = taxS->seen_X[k];
		ScoreMax[j] = taxS->scoreMat[k][j];
		}
	    taxS->scoreMat[k][j] = 0;
	    taxS->countMat[k][j] = 0;
	    }
	  
	  }

	  //printf("%d\t%f\n",uVec_X[i],ScoreVec[i]);
	  
	  if (TotalScoreMax[j] < ScoreVec[i])
	  {
	  TotalScoreMax[j] = ScoreVec[i];
	  TaxVec[j] = uniqueVec[i];
	  
	  FirstScore[j] = ScoreVec[i];
	  First_X[j] = uVec_X[i];
	  FirstWords[j] = CountVec[i];
	  
	  }
	}
	
	for (i=0; i<uPos; i++)
	{
	  if (ScoreVec[i] < FirstScore[j])
	  {
	    if (SecondScore[j] < ScoreVec[i])
	    {
	    SecondScore[j] = ScoreVec[i];
	    Second_X[j] = uVec_X[i];
	    SecondWords[j] = CountVec[i];
	    }
	  }
	  if (PRINTFLAG == 2)
	  {
	  printf("Tax:\t%d\t\tScore: %f\t\t%d\n", uniqueVec[i],ScoreVec[i],CountVec[i]);
	  }
	}
      //printf("\n");
      //printf("\tScore Max: %f\n",TotalScoreMax[j]);
      }
    
    //if (TotalScoreMax[8] > 0)
    //{
    //}
int bestTax[3];
double meanScoreFirst;
double meanScoreSecond;

bestTax[0] = bestTax[1] = bestTax[2] = 10010101;

int factor;
double minScore;
double maxloss = 0.6;
if (FirstWords[8] >= 6)
{
//printf("TaxID\tPrevious TaxID\tWords\tScore\tMean Score\n");
minScore = FirstScore[8]*maxloss;
    for (j=7; j>=0; j--)
    {   
//	break if score falls below threshold NEW!
      if (minScore > FirstScore[j])
      {
	    bestTax[0]= taxS->seqLength2;
	    bestTax[1]= First_X[j+1];
	    bestTax[2]= j+1;
	    break;
      }
      
      
      if (FirstWords[j] > 0 & SecondWords[j] > 0)
      {
	//printf("%d\t%d\n",FirstWords[j],SecondWords[j]);
	meanScoreFirst = FirstScore[j]/(double)FirstWords[j];
	meanScoreSecond = SecondScore[j]/(double)SecondWords[j];
	
	factor = floor(FirstWords[j]/SecondWords[j]);
	
	//printf("factor\n");
	  if (PRINTFLAG == 2)
	  {
	  printf("%d\t%d\t%d\t%f\t\t%f\n",TaxArray[First_X[j]][j],TaxArray[First_X[j]][j+1], FirstWords[j],FirstScore[j],meanScoreFirst);
	  printf("%d\t%d\t%d\t%f\t\t%f\n",TaxArray[Second_X[j]][j], TaxArray[Second_X[j]][j+1], SecondWords[j],SecondScore[j],meanScoreSecond);	  
	  }

	//printf("%d\n",factor);
	
    // Check if it is much more frequent than the 2. best hit.
	  if (factor >= 4)
	  {
	    if (TaxArray[First_X[j]][j+1] != TaxArray[First_X[j+1]][j+1])
	    {
	    bestTax[0]= taxS->seqLength2;
	    bestTax[1]= First_X[j+1];
	    bestTax[2]= j+1;
	    //printf("<<breaking at %d\t%d\t%d\n",j,TaxArray[First_X[j+1]][j+1],TaxArray[First_X[j]][j+1]);
	    break;
	    }
	  bestTax[0]= taxS->seqLength2;
	  bestTax[1]= First_X[j];
	  bestTax[2]= j;
	  }
	  
	  else
	  {
    // Check if the Score of the first hit is better...
	    if (meanScoreFirst <= meanScoreSecond)
	    {
	    bestTax[0]= taxS->seqLength2;
	    bestTax[1]= First_X[j+1];
	    bestTax[2]= j+1;
	    //printf(">breaking at %d\t%d\n",j,TaxArray[First_X[j+1]][j+1]);
	    //printf(">breaking at %d\t%d\n",j,First_X[j+1]);
	    break;
	    }
    //...or if the maximum changed...
	    if (TaxArray[First_X[j]][j+1] != TaxArray[First_X[j+1]][j+1])
	    {
	    bestTax[0]= taxS->seqLength2;
	    bestTax[1]= First_X[j+1];
	    bestTax[2]= j+1;
	    //printf("<breaking at %d\t%d\t%d\n",j,TaxArray[First_X[j+1]][j+1],TaxArray[First_X[j]][j+1]);
	    break;
	    }
    //...or a previous mean score was better...
	    if (meanScoreFirst < FirstScore[j+1]/(double)FirstWords[j+1] & TaxArray[First_X[j]][j] != 0)
	    {
	    bestTax[0]= taxS->seqLength2;
	    bestTax[1]= First_X[j+1];
	    bestTax[2]= j+1;
	    //printf("=breaking at %d\t%d\t%d\n",j,TaxArray[First_X[j+1]][j+1],TaxArray[First_X[j]][j+1]);
	    break;
	    }
	  bestTax[0]= taxS->seqLength2;
	  bestTax[1]= First_X[j];
	  bestTax[2]= j;
	  }
	}
      
	else if (FirstWords[j] > 0)
	{
	meanScoreFirst = FirstScore[j]/(double)FirstWords[j];
	    if (TaxArray[First_X[j]][j+1] != TaxArray[First_X[j+1]][j+1])
	    {
	    bestTax[0]= taxS->seqLength2;
	    bestTax[1]= First_X[j+1];
	    bestTax[2]= j+1;
	    //printf("<<breaking at %d\t%d\t%d\n",j,TaxArray[First_X[j+1]][j+1],TaxArray[First_X[j]][j+1]);
	    break;
	    }
    //...or a previous mean score was better...
	      if (meanScoreFirst < FirstScore[j+1]/(double)FirstWords[j+1] & TaxArray[First_X[j]][j] != 0)
	      {
	      bestTax[0]= taxS->seqLength2;
	      bestTax[1]= First_X[j+1];
	      bestTax[2]= j+1;
	      //printf("==breaking at %d\t%d\t%d\t\t%f\t%f\n",j,TaxArray[First_X[j+1]][j+1],TaxArray[First_X[j]][j+1],meanScoreFirst,FirstScore[j+1]/(double)FirstWords[j+1]);
	      break;
	      }
	  bestTax[0]= taxS->seqLength2;
	  bestTax[1]= First_X[j];
	  bestTax[2]= j;
	}
    //printf("\n");
    }
}

else
{
  
}
    
    
    //HACK
    struct sc value;
    union uproc_bst_key key;
    uproc_bstiter *iter;
    
    int FLAGG;
    
	
    iter = uproc_bstiter_create(scores);
    if (!iter) {
        return -1;
    }
    uproc_family aa,bb;
    while (!uproc_bstiter_next(iter, &key, &value, &aa, &bb)) {
    FLAGG = 0;
        uproc_family family = key.uint;
	//printf("trash\%u",family);
        double score = sc_finalize(&value);
	size_t seq_len = strlen(seq);
	if (pc->filter(seq, seq_len, family, score, pc->filter_arg))
	{
	//printf("<break>\n");
	FLAGG = 1;
	break;
	}
        
    
    }
    uproc_bstiter_destroy(iter);
    //HACK
    
    
    if (PRINTFLAG == 3)
    {    
    printf("%d\n",taxS->seqLength2);
    }
    
    if (FLAGG == 1 & PRINTFLAG == 1)
    {
    printf("%f,%d,%d,%d",TotalScoreMax[8],FirstWords[8],taxS->words,taxS->words_perfect);
      for (i=8; i>=0; i--)
      {
      //printf(",%f,%f", (double)(TotalScoreMax[i]/TotalScoreMax[8]*100),FirstScore[i]/(double)FirstWords[i]);
      printf(",%f", (double)(TotalScoreMax[i]/TotalScoreMax[8]*100));
      }
      
/*      for (i=8; i>=0; i--)
      {
	if (FirstWords[i] >= 1)
	{
	printf(",%f", FirstScore[i]/(double)FirstWords[i]);  
	}
	
	else
	{
	printf(",%f", 0.0);  
	}
      }
      
      for (i=8; i>=0; i--)
      {	
	if (SecondWords[i] >= 1)
	{
	printf(",%f", SecondScore[i]/(double)SecondWords[i]);    
	}
	
	else
	{
	printf(",%f", 0.0);  
	}
      }
      */


for (i=8; i>=0; i--)
      {
	if (FirstWords[i] >= 1)
	{
	printf(",%d", FirstWords[i]);  
	}
	
	else
	{
	printf(",%d", 0);  
	}
      }

      printf("\n");
      

      
      
    }

    //HACK
    
    
    
    
    
    
    
    
    res = scores_finalize(pc, seq, scores, *results, bestTax);
     
}

//Reset current thread-portion back to initialisation stage
taxS->seqLength = -1;
taxS->seqLength2 = 0;
taxS->used = 0;
taxS->words = 0;
taxS->words_perfect = 0;

    
error:
    uproc_bst_destroy(scores);
    return res;
}

void
uproc_protclass_set_trace(uproc_protclass *pc, uproc_protclass_trace_cb *cb,
                          void *cb_arg)
{
    pc->trace.cb = cb;
    pc->trace.cb_arg = cb_arg;
}

void
uproc_protresult_init(struct uproc_protresult *results)
{
    *results = (struct uproc_protresult) UPROC_PROTRESULT_INITIALIZER;
}

void
uproc_protresult_free(struct uproc_protresult *results)
{
    (void) results;
}

int
uproc_protresult_copy(struct uproc_protresult *dest,
                      const struct uproc_protresult *src)
{
    *dest = *src;
    return 0;
}
